/*
** Copyright (c) 2012, Romain Dura romain@shazbits.com
** 
** Permission to use, copy, modify, and/or distribute this software for any 
** purpose with or without fee is hereby granted, provided that the above 
** copyright notice and this permission notice appear in all copies.
** 
** THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES 
** WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF 
** MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR 
** IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/*
** Photoshop & misc math
** Blending modes, RGB/HSL/Contrast/Desaturate, levels control
**
** Romain Dura | Romz
** Blog: http://mouaif.wordpress.com
** Post: http://mouaif.wordpress.com/?p=94
*/



/*
** Desaturation
*/

vec4 Desaturate(vec3 color, float Desaturation)
{
	vec3 grayXfer = vec3(0.3, 0.59, 0.11);
	vec3 gray = CAST3(dot(grayXfer, color));
	return vec4(mix(color, gray, Desaturation), 1.0);
}


/*
** Hue, saturation, luminance
*/

vec3 RGBToHSL(vec3 color)
{
	vec3 hsl; // init to 0 to avoid warnings ? (and reverse if + remove first part)
	
	float fmin = min(min(color.r, color.g), color.b);    //Min. value of RGB
	float fmax = max(max(color.r, color.g), color.b);    //Max. value of RGB
	float delta = fmax - fmin;             //Delta RGB value

	hsl.z = (fmax + fmin) / 2.0; // Luminance

	if (delta == 0.0)		//This is a gray, no chroma...
	{
		hsl.x = 0.0;	// Hue
		hsl.y = 0.0;	// Saturation
	}
	else                                    //Chromatic data...
	{
		if (hsl.z < 0.5)
			hsl.y = delta / (fmax + fmin); // Saturation
		else
			hsl.y = delta / (2.0 - fmax - fmin); // Saturation
		
		float deltaR = (((fmax - color.r) / 6.0) + (delta / 2.0)) / delta;
		float deltaG = (((fmax - color.g) / 6.0) + (delta / 2.0)) / delta;
		float deltaB = (((fmax - color.b) / 6.0) + (delta / 2.0)) / delta;

		if (color.r == fmax )
			hsl.x = deltaB - deltaG; // Hue
		else if (color.g == fmax)
			hsl.x = (1.0 / 3.0) + deltaR - deltaB; // Hue
		else if (color.b == fmax)
			hsl.x = (2.0 / 3.0) + deltaG - deltaR; // Hue

		if (hsl.x < 0.0)
			hsl.x += 1.0; // Hue
		else if (hsl.x > 1.0)
			hsl.x -= 1.0; // Hue
	}

	return hsl;
}

float HueToRGB(float f1, float f2, float hue)
{
	if (hue < 0.0)
		hue += 1.0;
	else if (hue > 1.0)
		hue -= 1.0;
	float res;
	if ((6.0 * hue) < 1.0)
		res = f1 + (f2 - f1) * 6.0 * hue;
	else if ((2.0 * hue) < 1.0)
		res = f2;
	else if ((3.0 * hue) < 2.0)
		res = f1 + (f2 - f1) * ((2.0 / 3.0) - hue) * 6.0;
	else
		res = f1;
	return res;
}

vec3 HSLToRGB(vec3 hsl)
{
	vec3 rgb;
	
	if (hsl.y == 0.0)
		rgb = CAST3(hsl.z); // Luminance
	else
	{
		float f2;
		
		if (hsl.z < 0.5)
			f2 = hsl.z * (1.0 + hsl.y);
		else
			f2 = (hsl.z + hsl.y) - (hsl.y * hsl.z);
			
		float f1 = 2.0 * hsl.z - f2;
		
		rgb.r = HueToRGB(f1, f2, hsl.x + (1.0/3.0));
		rgb.g = HueToRGB(f1, f2, hsl.x);
		rgb.b= HueToRGB(f1, f2, hsl.x - (1.0/3.0));
	}
	
	return rgb;
}


/*
** Contrast, saturation, brightness
** Code of this function is from TGM's shader pack
** http://irrlicht.sourceforge.net/phpBB2/viewtopic.php?t=21057
*/

vec3 ContrastSaturationBrightness(vec3 color, float brt, float sat, float con)
{
	// Increase or decrease theese values to adjust r, g and b color channels seperately
	const float AvgLumR = 0.5;
	const float AvgLumG = 0.5;
	const float AvgLumB = 0.5;
	
	const vec3 LumCoeff = vec3(0.2125, 0.7154, 0.0721);
	
	vec3 AvgLumin = vec3(AvgLumR, AvgLumG, AvgLumB);
	vec3 brtColor = color * brt;
	vec3 intensity = CAST3(dot(brtColor, LumCoeff));
	vec3 satColor = mix(intensity, brtColor, sat);
	vec3 conColor = mix(AvgLumin, satColor, con);
	return conColor;
}


/*
** Float blending modes
** Adapted from here: http://www.nathanm.com/photoshop-blending-math/
** But I modified the HardMix (wrong condition), Overlay, SoftLight, ColorDodge, ColorBurn, VividLight, PinLight (inverted layers) ones to have correct results
*/

#define BlendLinearDodgef 			BlendAddf
#define BlendLinearBurnf 			BlendSubstractf
#define BlendAddf(base, blend) 		min(base + blend, 1.0)
#define BlendSubstractf(base, blend) 	max(base + blend - 1.0, 0.0)
#define BlendLightenf(base, blend) 		max(blend, base)
#define BlendDarkenf(base, blend) 		min(blend, base)
#define BlendLinearLightf(base, blend) 	(blend < 0.5 ? BlendLinearBurnf(base, (2.0 * blend)) : BlendLinearDodgef(base, (2.0 * (blend - 0.5))))
#define BlendScreenf(base, blend) 		(1.0 - ((1.0 - base) * (1.0 - blend)))
#define BlendOverlayf(base, blend) 	(base < 0.5 ? (2.0 * base * blend) : (1.0 - 2.0 * (1.0 - base) * (1.0 - blend)))
#define BlendSoftLightf(base, blend) 	((blend < 0.5) ? (2.0 * base * blend + base * base * (1.0 - 2.0 * blend)) : (sqrt(base) * (2.0 * blend - 1.0) + 2.0 * base * (1.0 - blend)))
#define BlendColorDodgef(base, blend) 	((blend == 1.0) ? blend : min(base / (1.0 - blend), 1.0))
#define BlendColorBurnf(base, blend) 	((blend == 0.0) ? blend : max((1.0 - ((1.0 - base) / blend)), 0.0))
#define BlendVividLightf(base, blend) 	((blend < 0.5) ? BlendColorBurnf(base, (2.0 * blend)) : BlendColorDodgef(base, (2.0 * (blend - 0.5))))
#define BlendPinLightf(base, blend) 	((blend < 0.5) ? BlendDarkenf(base, (2.0 * blend)) : BlendLightenf(base, (2.0 *(blend - 0.5))))
#define BlendHardMixf(base, blend) 	((BlendVividLightf(base, blend) < 0.5) ? 0.0 : 1.0)
#define BlendReflectf(base, blend) 		((blend == 1.0) ? blend : min(base * base / (1.0 - blend), 1.0))


/*
** Vector3 blending modes
*/

// Component wise blending
//#define Blend(base, blend, funcf) 		vec3(funcf(base.r, blend.r), funcf(base.g, blend.g), funcf(base.b, blend.b))
// vec3((base.r, blend.r), (base.g, blend.g), (base.b, blend.b))

#define BlendNormal(base, blend) 		(blend)
#define BlendLighten				BlendLightenf
#define BlendDarken				BlendDarkenf
#define BlendMultiply(base, blend) 		(base * blend)
#define BlendAverage(base, blend) 		((base + blend) / 2.0)
#define BlendAdd(base, blend) 		min(base + blend, CAST3(1.0))
#define BlendSubstract(base, blend) 	max(base + blend - CAST3(1.0), CAST3(0.0))
#define BlendDifference(base, blend) 	abs(base - blend)
#define BlendNegation(base, blend) 	(CAST3(1.0) - abs(CAST3(1.0) - base - blend))
#define BlendExclusion(base, blend) 	(base + blend - 2.0 * base * blend)
#define BlendScreen(base, blend) 		vec3(BlendScreenf(base.r, blend.r), BlendScreenf(base.g, blend.g), BlendScreenf(base.b, blend.b))
#define BlendOverlay(base, blend) 		vec3(BlendOverlayf(base.r, blend.r), BlendOverlayf(base.g, blend.g), BlendOverlayf(base.b, blend.b))
#define BlendSoftLight(base, blend) 	vec3(BlendSoftLightf(base.r, blend.r), BlendSoftLightf(base.g, blend.g), BlendSoftLightf(base.b, blend.b))
#define BlendHardLight(base, blend) 	BlendOverlay(blend, base)
#define BlendColorDodge(base, blend) 	vec3(BlendColorDodgef(base.r, blend.r), BlendColorDodgef(base.g, blend.g), BlendColorDodgef(base.b, blend.b))
#define BlendColorBurn(base, blend) 	vec3(BlendColorBurnf(base.r, blend.r), BlendColorBurnf(base.g, blend.g), BlendColorBurnf(base.b, blend.b))
#define BlendLinearDodge			BlendAdd
#define BlendLinearBurn			BlendSubstract
// Linear Light is another contrast-increasing mode
// If the blend color is darker than midgray, Linear Light darkens the image by decreasing the brightness. If the blend color is lighter than midgray, the result is a brighter image due to increased brightness.
#define BlendLinearLight(base, blend) 	vec3(BlendLinearLightf(base.r, blend.r), BlendLinearLightf(base.g, blend.g), BlendLinearLightf(base.b, blend.b))
#define BlendVividLight(base, blend) 	vec3(BlendVividLightf(base.r, blend.r), BlendVividLightf(base.g, blend.g), BlendVividLightf(base.b, blend.b))
#define BlendPinLight(base, blend) 		vec3(BlendPinLightf(base.r, blend.r), BlendPinLightf(base.g, blend.g), BlendPinLightf(base.b, blend.b))
#define BlendHardMix(base, blend) 		vec3(BlendHardMixf(base.r, blend.r), BlendHardMixf(base.g, blend.g), BlendHardMixf(base.b, blend.b))
#define BlendReflect(base, blend) 		vec3(BlendReflectf(base.r, blend.r), BlendReflectf(base.g, blend.g), BlendReflectf(base.b, blend.b))
#define BlendGlow(base, blend) 		BlendReflect(blend, base)
#define BlendPhoenix(base, blend) 		(min(base, blend) - max(base, blend) + CAST3(1.0))

#define BlendOpacity(base, blend, F, O) 	mix(base, F(base, blend), O)
//#define BlendOpacity(base, blend, F, O) 	(F(base, blend) * O + base * (1.0 - O))

#define BlendTint(base, blend)		(CAST3(max(base.x, max(base.y, base.z))) * blend)

// Hue Blend mode creates the result color by combining the luminance and saturation of the base color with the hue of the blend color.
vec3 BlendHue(vec3 base, vec3 blend)
{
	vec3 baseHSL = RGBToHSL(base);
	return HSLToRGB(vec3(RGBToHSL(blend).r, baseHSL.g, baseHSL.b));
}

// Saturation Blend mode creates the result color by combining the luminance and hue of the base color with the saturation of the blend color.
vec3 BlendSaturation(vec3 base, vec3 blend)
{
	vec3 baseHSL = RGBToHSL(base);
	return HSLToRGB(vec3(baseHSL.r, RGBToHSL(blend).g, baseHSL.b));
}

// Color Mode keeps the brightness of the base color and applies both the hue and saturation of the blend color.
vec3 BlendColor(vec3 base, vec3 blend)
{
	vec3 blendHSL = RGBToHSL(blend);
	return HSLToRGB(vec3(blendHSL.r, blendHSL.g, RGBToHSL(base).b));
}

// Luminosity Blend mode creates the result color by combining the hue and saturation of the base color with the luminance of the blend color.
vec3 BlendLuminosity(vec3 base, vec3 blend)
{
	vec3 baseHSL = RGBToHSL(base);
	return HSLToRGB(vec3(baseHSL.r, baseHSL.g, RGBToHSL(blend).b));
}


/*
** Gamma correction
** Details: http://blog.mouaif.org/2009/01/22/photoshop-gamma-correction-shader/
*/

#define GammaCorrection(color, gamma)								pow(color, 1.0 / gamma)

/*
** Levels control (input (+gamma), output)
** Details: http://blog.mouaif.org/2009/01/28/levels-control-shader/
*/

vec3 ApplyBlending(const int blendMode, in vec3 A, in vec3 B, in float opacity)
{
	// Must assume that blendMode == BLENDMODE!!!
	// The if/else construct below leads to an memory exhausted error otherwise
	
#if BLENDMODE == 1
		return BlendOpacity(A, B, BlendDarken, opacity);
#endif
#if BLENDMODE == 2
		return BlendOpacity(A, B, BlendMultiply, opacity);
#endif
#if BLENDMODE == 3
		return BlendOpacity(A, B, BlendColorBurn, opacity);
#endif
#if BLENDMODE == 4
		return BlendOpacity(A, B, BlendLinearBurn, opacity);
#endif
#if BLENDMODE == 5
		return min(A, B);
#endif
#if BLENDMODE == 6
		return BlendOpacity(A, B, BlendLighten, opacity);
#endif
#if BLENDMODE == 7
		return BlendOpacity(A, B, BlendScreen, opacity);
#endif
#if BLENDMODE == 8
		return BlendOpacity(A, B, BlendColorDodge, opacity);
#endif
#if BLENDMODE == 9
		return BlendOpacity(A, B, BlendLinearDodge, opacity);
#endif
#if BLENDMODE == 10
		return max(A, B);
#endif
#if BLENDMODE == 11
		return BlendOpacity(A, B, BlendOverlay, opacity);
#endif
#if BLENDMODE == 12
		return BlendOpacity(A, B, BlendSoftLight, opacity);
#endif
#if BLENDMODE == 13
		return BlendOpacity(A, B, BlendHardLight, opacity);
#endif
#if BLENDMODE == 14
		return BlendOpacity(A, B, BlendVividLight, opacity);
#endif
#if BLENDMODE == 15
		return BlendOpacity(A, B, BlendLinearLight, opacity);
#endif
#if BLENDMODE == 16
		return BlendOpacity(A, B, BlendPinLight, opacity);
#endif
#if BLENDMODE == 17
		return BlendOpacity(A, B, BlendHardMix, opacity);
#endif
#if BLENDMODE == 18
		return BlendOpacity(A, B, BlendDifference, opacity);
#endif
#if BLENDMODE == 19
		return BlendOpacity(A, B, BlendExclusion, opacity);
#endif
#if BLENDMODE == 20
		return BlendOpacity(A, B, BlendSubstract, opacity);
#endif
#if BLENDMODE == 21
		return BlendOpacity(A, B, BlendReflect, opacity);
#endif
#if BLENDMODE == 22
		return BlendOpacity(A, B, BlendGlow, opacity);
#endif
#if BLENDMODE == 23
		return BlendOpacity(A, B, BlendPhoenix, opacity);
#endif
#if BLENDMODE == 24
		return BlendOpacity(A, B, BlendAverage, opacity);
#endif
#if BLENDMODE == 25
		return BlendOpacity(A, B, BlendNegation, opacity);
#endif
#if BLENDMODE == 26
		return BlendOpacity(A, B, BlendHue, opacity);
#endif
#if BLENDMODE == 27
		return BlendOpacity(A, B, BlendSaturation, opacity);
#endif
#if BLENDMODE == 28
		return BlendOpacity(A, B, BlendColor, opacity);
#endif
#if BLENDMODE == 29
		return BlendOpacity(A, B, BlendLuminosity, opacity);
#endif
#if BLENDMODE == 30
		return BlendOpacity(A, B, BlendTint, opacity);
#endif

	//if (blendMode == 1)
	//	return BlendOpacity(A, B, BlendDarken, opacity);
	//else if (blendMode == 2)
	//	return BlendOpacity(A, B, BlendMultiply, opacity);
	//else if (blendMode == 3)
	//	return BlendOpacity(A, B, BlendColorBurn, opacity);
	//else if (blendMode == 4)
	//	return BlendOpacity(A, B, BlendLinearBurn, opacity);
	//else if (blendMode == 5)
	//	return min(A, B); // darker color
	//else if (blendMode == 6)
	//	return BlendOpacity(A, B, BlendLighten, opacity);
	//else if (blendMode == 7)
	//	return BlendOpacity(A, B, BlendScreen, opacity);
	//else if (blendMode == 8)
	//	return BlendOpacity(A, B, BlendColorDodge, opacity);
	//else if (blendMode == 9)
	//	return BlendOpacity(A, B, BlendLinearDodge, opacity);
	//else if (blendMode == 10)
	//	return max(A, B); // lighter color
	//else if (blendMode == 11)
	//	return BlendOpacity(A, B, BlendOverlay, opacity);
	//else if (blendMode == 12)
	//	return BlendOpacity(A, B, BlendSoftLight, opacity);
	//else if (blendMode == 13)
	//	return BlendOpacity(A, B, BlendHardLight, opacity);
	//else if (blendMode == 14)
	//	return BlendOpacity(A, B, BlendVividLight, opacity);
	//else if (blendMode == 15)
	//	return BlendOpacity(A, B, BlendLinearLight, opacity);
	//else if (blendMode == 16)
	//	return BlendOpacity(A, B, BlendPinLight, opacity);
	//else if (blendMode == 17)
	//	return BlendOpacity(A, B, BlendHardMix, opacity);
	//else if (blendMode == 18)
	//	return BlendOpacity(A, B, BlendDifference, opacity);
	//else if (blendMode == 19)
	//	return BlendOpacity(A, B, BlendExclusion, opacity);
	//else if (blendMode == 20)
	//	return BlendOpacity(A, B, BlendSubstract, opacity);
	//else if (blendMode == 21)
	//	return BlendOpacity(A, B, BlendReflect, opacity);
	//else if (blendMode == 22)
	//	return BlendOpacity(A, B, BlendGlow, opacity);
	//else if (blendMode == 23)
	//	return BlendOpacity(A, B, BlendPhoenix, opacity);
	//else if (blendMode == 24)
	//	return BlendOpacity(A, B, BlendAverage, opacity);
	//else if (blendMode == 25)
	//	return BlendOpacity(A, B, BlendNegation, opacity);
	//else if (blendMode == 26)
	//	return BlendOpacity(A, B, BlendHue, opacity);
	//else if (blendMode == 27)
	//	return BlendOpacity(A, B, BlendSaturation, opacity);
	//else if (blendMode == 28)
	//	return BlendOpacity(A, B, BlendColor, opacity);
	//else if (blendMode == 29)
	//	return BlendOpacity(A, B, BlendLuminosity, opacity);
	//else if (blendMode == 30)
	//	return BlendOpacity(A, B, BlendTint, opacity);
	//else
		return BlendOpacity(A, B, BlendNormal, opacity);
}
